
namespace eval Layers {
    variable SelectColumn ;#local temporary value
    variable _tree_isopen  ;#to save/restore the state open/collapse of the tree
}

proc Layers::EncodeName { x } {
    return [binary encode base64 [encoding convertto utf-8 $x]]
}

proc Layers::DecodeName { x } {
    return [encoding convertfrom utf-8  [binary decode base64 $x]]
}

proc Layers::SetLayerAsSelection { T layer } {
    if { ![winfo exists $T] } { return }
    $T selection clear
    set item [$T item id "tag [Layers::EncodeName $layer]"]
    if { $item != "" } {
        $T selection add $item
        $T see $item
    }
}

proc Layers::SelectLayersByRegexpPressOk { T e } {
    global wChangeLay LayerSelRegexp LayerSelAdd LayerSelCase
    if { $LayerSelAdd == "substitute" } {
        $T selection clear
    }
    foreach item [$T item range 1 end] {
        #set fullname [Layers::DecodeName [$T item tag names $item]]
        set i [$T item text $item 0]
        set change ""
        if { $LayerSelRegexp != "regex" } {
            if { $LayerSelCase != "case" } {
                if { [string match [string tolower [$e get]] \
                    [string tolower $i]] } { set change yes }
                } else {
                if { [string match [$e get] $i] } {
                    set change yes
                }
            }
        } else {
            if { $LayerSelCase != "case" } {
                if { [regexp -nocase -- [$e get] $i res] } {
                    if { $i == $res } { set change yes }
                }
            } else {
                if { [regexp -- [$e get] $i res] } {
                    if { $i == $res } { set change yes }
                }
            }
        }
        if { $change == "yes" } {
            if { $LayerSelAdd != "substract" } {
                $T selection add $item
            } else {
                $T selection clear $item
            }
        }
    }
    destroy $wChangeLay.sel
    if { [winfo exists $wChangeLay.ent.e] } {
        focus $wChangeLay.ent.e
    }
}

proc Layers::SelectLayersByRegexp { tbl } {
    global wChangeLay LayerSelRegexp LayerSelAdd LayerSelCase

    if { [winfo exists $wChangeLay.sel] } {
        destroy $wChangeLay.sel
    }
    ttk::frame $wChangeLay.sel -style ridge.TFrame -borderwidth 4
    ttk::frame $wChangeLay.sel.up
    set e [ttk::entry $wChangeLay.sel.up.e]
    ttk::button $wChangeLay.sel.up.ok -text [_ "OK"] \
        -command [list Layers::SelectLayersByRegexpPressOk $tbl $e]
    grid $e $wChangeLay.sel.up.ok -sticky ew
    grid configure $wChangeLay.sel.up.ok -sticky w
    grid columnconfigure $wChangeLay.sel.up 0 -weight 1

    ttk::frame $wChangeLay.sel.down -borderwidth 2
    if { ![info exists LayerSelRegexp] } { set LayerSelRegexp 0 }
    ttk::checkbutton $wChangeLay.sel.down.reg -text [_ "Regexp"] -onvalue regex \
        -variable LayerSelRegexp
    if { ![info exists LayerSelCase] } { set LayerSelCase 0 }
    ttk::checkbutton $wChangeLay.sel.down.case -text [_ "Case"] -onvalue case \
        -variable LayerSelCase

    ttk::frame $wChangeLay.sel.rad -borderwidth 2
    ttk::radiobutton $wChangeLay.sel.rad.substitute -text [_ "Substitute"] -value substitute \
        -variable LayerSelAdd
    ttk::radiobutton $wChangeLay.sel.rad.add -text [_ "Add"] -value add \
        -variable LayerSelAdd
    ttk::radiobutton $wChangeLay.sel.rad.substract -text [_ "Substract"] -value substract \
        -variable LayerSelAdd

    set LayerSelRegexp ""
    set LayerSelAdd substitute
    set LayerSelCase ""

    grid $wChangeLay.sel.down.reg $wChangeLay.sel.down.case -sticky ew

    grid $wChangeLay.sel.rad.substitute $wChangeLay.sel.rad.add \
        $wChangeLay.sel.rad.substract -sticky ew

    grid $wChangeLay.sel.up -sticky ew
    grid $wChangeLay.sel.down -sticky ew
    grid $wChangeLay.sel.rad -sticky ew

    grid $wChangeLay.sel -row 1 -sticky new
    grid columnconfigure $wChangeLay.sel 0 -weight 1

    focus $e
    bind $e <Return> [list $wChangeLay.sel.up.ok invoke]
}

proc Layers::ListEntitiesSelection { T } {
    Layers::IfNothingSelectedSelectAll $T
    set layer_names [Layers::GetSelectedLayernames $T]
    set text [Layers::ListEntities $layer_names]
    W $text [_ "Layer's entities"]
}

#limit 43000 is a trick to avoid long lines >=43684 characters are not visible in current Tk text widget!!
#if limit==0 then consider as no limit
proc Layers::ListEntities { layer_names {limit 43000}} {
    set text ""
    set view_mode [GiD_Info Project ViewMode]
    if { $view_mode == "GEOMETRYUSE"}  {
        set overs [GetGeometryCommands]
        set types [GetGeometryLabels]
    } elseif { $view_mode == "MESHUSE"}  {
        set overs [GetMeshCommands]
        set types [GetMeshLabels]
    } else {
        error "Unexpected mode $view_mode"
    }
    foreach fullname $layer_names {
        set has_entities 0
        foreach over $overs type $types {
            set num_entities [GiD_Info layers -count -entities $over $fullname]
            if { $num_entities > 0 } {
                set has_entities 1
                break
            }
        }
        append text "$fullname:\n"
        if { $has_entities } {
            foreach over $overs type $types {
                set type_data [GiD_Info layers -entities $over $fullname]
                if { [llength $type_data] > 0 } {
                    if { $type == [_ "Elements"] } {
                        #show also the number of types of the elements
                        unset -nocomplain num_element_type
                        foreach element_id $type_data {
                            incr num_element_type([lindex [GiD_Mesh get element $element_id] 1])
                        }
                        set element_names ""
                        foreach element_type [lsort -dictionary [array names num_element_type]] {
                            lappend element_names [list $num_element_type($element_type) [_ $element_type]]
                        }
                        set count [llength $type_data]
                        if { $limit && [string length $type_data] > $limit} {
                            #long lines >=43684 characters are not visible in current Tk text widget!!
                            set type_data [string range $type_data 0 $limit]...
                        }
                        append text "  $count $type ([join $element_names {, }]): $type_data\n"
                    } else {
                        set count [llength $type_data]
                        if { $limit && [string length $type_data] > $limit} {
                            #long lines >=43684 characters are not visible in current Tk text widget!!
                            set type_data [string range $type_data 0 $limit]...
                        }
                        append text "  $count $type: $type_data\n"
                    }
                }
            }
            append text "\n"
        } else {
            append text "\n"
        }
    }
    return $text
}

proc Layers::FillInfo { } {
    global wChangeLay GidPriv
    if { [info exists wChangeLay] } {
        Layers::FillTableInfoLayers $GidPriv(Layers,table)
        if { [winfo exists $wChangeLay.touse.l] } {
            $wChangeLay.touse.l configure -text [GiD_Info Project LayerToUse]
        }
    }
    if { [info exists GidPriv(ComboLayers,table)] } {
        Layers::FillTableInfoLayers $GidPriv(ComboLayers,table)
        #update size
        set_layers_menu_size $GidPriv(ComboLayers,table) [winfo toplevel $GidPriv(ComboLayers,table)]
    }
    if { [info exists GidPriv(ComboLayers,entry)] && [winfo exists $GidPriv(ComboLayers,entry)] } {
        $GidPriv(ComboLayers,entry) delete 0 end
        $GidPriv(ComboLayers,entry) insert end [GiD_Info Project LayerToUse]
    }
}

proc Layers::SetLayersToDelete { tbl } {
    set parent [winfo parent $tbl]
    set LayersSelection [Layers::GetSelectedLayernames $tbl]
    if { ![llength $LayersSelection] } {
        WarnWin [_ "before applying operation delete it is necessary to select layers"]
    } else {
        set ask 1
        set ret 0
        set delete_only_empty_layers 1
        GiD_Process 'Layers
        foreach i $LayersSelection {
            if { [lindex [GiD_Info Layers -canbedeleted $i] 0] == 0 } {
                if { $ask } {
                    set answer [MessageBoxOptionsButtons  [_ "Question"] [_ "Confirm that you want to delete layer '%s' and its entities?" $i] \
                        {0 1 2} [list [_ "Yes"] [_ "No"] [_ "Cancel"]] question [_ "Do it for all"]]
                    set do_it_for_all 0
                    lassign $answer ret do_it_for_all
                    if { $do_it_for_all } {
                        set ask 0
                    }
                }
                if { $ret == 0 } {
                    #yes
                    if { [GiD_Layers exists $i] } {
                        GiD_Process Delete $i Yes
                    }
                } elseif { $ret == 1 } {
                    #no
                } elseif { $ret == 2 } {
                    #cancel
                    break
                } else {
                    #unexpected
                    break
                }
            } else {
                if { [GiD_Layers exists $i] } {
                    GiD_Process Delete $i
                }
            }
        }
        GiD_Process escape
    }
}

proc Layers::SetLayersToColor { tbl } {
    set parent [winfo parent $tbl]
    set LayersSelection [Layers::GetSelectedLayernames $tbl]
    if { $LayersSelection == "" } {
        WarnWin [_ "before applying operation color it is necessary to select layers"]
        return
    }
    set sel $LayersSelection ;#trick because GiD_Process 'Layers can change LayersSelection

    GiD_Process 'Layers
    set LayersSelection $sel
    set layer_name [lindex $LayersSelection 0]
    set cur_col [GiD_Layers get color $layer_name]
    #set retcol [tk_chooseColor -parent $parent -title [_ "Select color"] -initialcolor $cur_col]
    set retcol [GIDChooseColor $parent.selectcolor -title [_ "Select color"] -color $cur_col]
    if { $retcol != "" } {
        set retcol [string range $retcol 1 end]
        if { [string length $retcol] == 8 } {
            scan $retcol %2x%2x%2x%2x col1 col2 col3 alpha
            set col [format %03i%03i%03i $col1 $col2 $col3]
            foreach i $LayersSelection {
                GiD_Process Color $i $col Transparent $i $alpha
            }
        } else {
            scan $retcol %2x%2x%2x col1 col2 col3
            set col [format %03i%03i%03i $col1 $col2 $col3]
            foreach i $LayersSelection {
                GiD_Process Color $i $col
            }
        }
    }
    GiD_Process escape
}

proc Layers::SetLayersTo { type tbl {newname ""} } {
    set LayersSelection [Layers::GetSelectedLayernames $tbl]
    if { $LayersSelection == "" } {
        WarnWin [_ "before applying operation %s it is necessary to select layers" $type]
        return
    }

    set VarToSend 'Layers
    foreach i $LayersSelection {
        switch $type {
            ON {
                lappend VarToSend On $i
            }
            OFF {
                lappend VarToSend Off $i
            }
            FREEZE {
                lappend VarToSend Freeze $i
            }
            UNFREEZE {
                lappend VarToSend UnFreeze $i
            }
            TRANSPARENT {
                lappend VarToSend Transparent $i 127
            }
            OPAQUE {
                lappend VarToSend Opaque $i
            }
            TOUSE {
                lappend VarToSend ToUse $i
            }
            RENAME {
                lappend VarToSend ChangeName $i $newname
            }
            BRINGTOFRONT {
                lappend VarToSend BringToFront $i
            }
        }
    }
    lappend VarToSend escape

    GiD_Process {*}$VarToSend

    #Layers::FillInfo ;#automatically called
    if { $type == "RENAME" } {
        #Trick to have selected the new name after Layers::FillInfo will be automatically called
        Layers::SetLayerAsSelection $tbl $newname
    }
}

proc Layers::SendLayerEntitiesAA { T what } {
    set LayersSelection [Layers::GetSelectedLayernames $T]
    if { [llength $LayersSelection] !=1 } {
        WarnWin [_ "before applying 'Entities' it is necessary to select 1 layer"]
        return
    }
    set layer [lindex $LayersSelection 0]
    set command [list 'Layers Entities $layer]
    if { [GiD_Set LayersAlsoLower] } {
        lappend command LowerEntities
    }
    if { [GiD_Set LayersAlsoHigher] } {
        lappend command HigherEntities
    }
    lappend command $what
    GiD_Process {*}$command
    if { [info exists ::wChangeLayButtonFrame] && [winfo exists $::wChangeLayButtonFrame] } {
        set SmallWinSelecting [GiD_Set SmallWinSelecting]
        set w [winfo parent $::wChangeLayButtonFrame]
        FinishButton $w $::wChangeLayButtonFrame [_ "Press 'Finish' to end selection"] "" disableall  $SmallWinSelecting
    }
}

proc Layers::SendLayerEntities { m T } {
    $m delete 0 end
    global GidPriv
    set ::GidPriv(LayersAlsoLower) [GiD_Set LayersAlsoLower]
    set ::GidPriv(LayersAlsoHigher) [GiD_Set LayersAlsoHigher]
    $m add checkbutton -label [_ "Also lower entities"] -variable ::GidPriv(LayersAlsoLower) -command {GiD_Set LayersAlsoLower $::GidPriv(LayersAlsoLower)}
    $m add checkbutton -label [_ "Also higher entities"] -variable ::GidPriv(LayersAlsoHigher) -command {GiD_Set LayersAlsoHigher $::GidPriv(LayersAlsoHigher)}

    set geom_commands [GetGeometryCommands]
    set geom_labels [GetGeometryLabels]
    set geom_images [GetGeometryImageFiles]
    lappend geom_commands {*}{Dimensions All}
    lappend geom_labels {*}[list [_ "Dimensions"] [_ "All"]]
    lappend geom_images {*}{dimension.png alltypes.png}

    set mesh_commands [GetMeshCommands]
    set mesh_labels [GetMeshLabels]
    set mesh_images [GetMeshImageFiles]
    lappend mesh_commands {*}{Dimensions All}
    lappend mesh_labels {*}[list [_ "Dimensions"] [_ "All"]]
    lappend mesh_images {*}{dimension.png alltypes_mesh.png}

    AddEntitiesToMenuAfter 1 $m [list Layers::SendLayerEntitiesAA $T] \
        [list $geom_commands $geom_labels $geom_images] \
        [list $mesh_commands $mesh_labels $mesh_images]

    $m add separator
    $m add command -label [_ "Self-dependencies"] -command [list Layers::SendToSelfDependenciesSelection $T]
}

proc Layers::LayerToUseEntitiesAA { what } {
    GidUtils::SetWarnLine [_ "Pick one entity to set its layer as current layer to use"]
    GidUtils::DisableWarnLine
    set nnum [GidUtils::PickEntities $what single]
    GidUtils::EnableWarnLine
    if { $nnum != "" } {
        set retval [GiD_Info list_entities $what $nnum]
        set layername ""
        regexp {LAYER:([^\n\r]+)} $retval {} layername
        set layername [string trim $layername]
        if { $layername != "" } {
            GiD_Process 'Layers ToUse $layername
            Layers::FillInfo
            Layers::SetLayerAsSelection $::GidPriv(Layers,table) $layername
        } else {
            WarnWin [_ "Selected entity does not belong to one layer"]
        }
    }
}

proc Layers::SendToSelfDependencies { layer_name } {
    set changes 0
    foreach over {volume surface line element} {
        set ids [GiD_EntitiesLayers get $layer_name ${over}s]
        #set ids [GiD_Geometry list -layer $layer_name $over]
        if { [objarray length $ids] } {
            GiD_Process 'Layers Entities $layer_name LowerEntities [string totitle ${over}s] {*}$ids escape
            incr changes
        }
    }
    return $changes
}

proc Layers::SendToSelfDependenciesSelection { T } {
    set layer_names [Layers::GetSelectedLayernames $T]
    set num_selected_layers [llength $layer_names]
    if { $num_selected_layers } {
        set changes 0
        GidUtils::DisableGraphics ;#to avoid refresh LAYER several times
        foreach layer_name $layer_names {
            incr changes [Layers::SendToSelfDependencies $layer_name]
        }
        GidUtils::EnableGraphics
        if { $changes } {
            GiD_Redraw
        }
    }
    GidUtils::SetWarnLine [_ "Sent dependent entities of %s layers" $num_selected_layers]
}

proc Layers::LayerToUseEntities { menu } {
    AddEntitiesToMenuXXX $menu Layers::LayerToUseEntitiesAA \
        [list [GetGeometryCommands] [GetGeometryLabelsSingular] [GetGeometryImageFiles]] \
        [list [GetMeshCommands] [GetMeshLabelsSingular]s [GetMeshImageFiles]]
}

proc Layers::SendLayerEntitiesToBackAA { what } {
    set command 'Layers
    if { $::GidPriv(LayersBackOpposite) } {
        lappend command SendToBackOpposite
    } else {
        lappend command SendToBack
    }
    if { [GiD_Set LayersAlsoLower] } {
        lappend command LowerEntities
    }
    if { [GiD_Set LayersAlsoHigher] } {
        lappend command HigherEntities
    }
    lappend command $what
    GiD_Process {*}$command
    if { [info exists ::wChangeLayButtonFrame] && [winfo exists $::wChangeLayButtonFrame] } {
        set SmallWinSelecting [GiD_Set SmallWinSelecting]
        set w [winfo parent $::wChangeLayButtonFrame]
        FinishButton $w $::wChangeLayButtonFrame [_ "Press 'Finish' to end selection"] Layers::FillInfo disableall $SmallWinSelecting
    }
}

proc Layers::SendLayerEntitiesToBack { m tbl } {
    global GidPriv
    $m delete 0 end
    set ::GidPriv(LayersAlsoLower) [GiD_Set LayersAlsoLower]
    set ::GidPriv(LayersAlsoHigher) [GiD_Set LayersAlsoHigher]
    $m add checkbutton -label [_ "Also lower entities"] -variable ::GidPriv(LayersAlsoLower) -command {GiD_Set LayersAlsoLower $::GidPriv(LayersAlsoLower)}
    $m add checkbutton -label [_ "Also higher entities"] -variable ::GidPriv(LayersAlsoHigher) -command {GiD_Set LayersAlsoHigher $::GidPriv(LayersAlsoHigher)}
    $m add checkbutton -label [_ "Opposite"] -variable GidPriv(LayersBackOpposite)

    AddEntitiesToMenuAfter 2 $m Layers::SendLayerEntitiesToBackAA \
        [list [GetGeometryCommands 1] [GetGeometryLabels 1] [GetGeometryImageFiles 1]] \
        [list [GetMeshCommands 1] [GetMeshLabels 1] [GetMeshImageFiles 1]]

    $m add separator
    if { [GiD_Info layers -hasbacklayers] } {
        $m add command -label [_ "Bring to front"] \
            -command [list Layers::SetLayersTo BRINGTOFRONT $tbl] \
            -image [gid_themes::GetImage bring_to_front.png small_icons] -compound left
        $m add command -label [_ "Bring all to front"] \
            -command {GiD_Process 'Layers BringToFrontAll escape ; Layers::FillInfo} \
            -image [gid_themes::GetImage bring_to_front.png small_icons] -compound left
    } else {
        $m add command -label [_ "Bring to front"] -state disabled \
            -image [gid_themes::GetImage bring_to_front.png small_icons] -compound left
        $m add command -label [_ "Bring all to front"] -state disabled \
            -image [gid_themes::GetImage bring_to_front.png small_icons] -compound left
    }
}

proc Layers::CreateNewLayerFolder { {T ""} {name ""} } {
    set prefix ""
    if { $T != "" && [$T selection count] == 1 } {
        set item [$T selection get 0]
        set prefix [Layers::DecodeName [$T item tag names $item]]
    } else {
        WarnWin [_ "A parent layer must be selected before"]
        return ""
    }
    if { $name == "" } {
        set name [Layers::GetAutomaticLayerName $prefix]
    } else {
        if { [GiD_Layers is_forbidden_name $name] } {
            WarnWin [_ "Bad layer name"]
            return ""
        }
    }
    GiD_Process 'Layers New $name escape
    update ;#to fill in again the treectrl to create the item
    set item [$T item id "tag [Layers::EncodeName $name]"]
    if { $item != "" } {
        $T activate $item
        Layers::BeginEditLayers $T
    }
    return $name
}

proc Layers::NameJoin { args } {
    set parts [list]
    foreach item $args {
        if { $item != "" } {
            lappend parts $item
        }
    }
    if { [llength $parts] > 1 } {
        set fullname [join $parts //]
    } else {
        #join to remove {} when item has spaces
        set fullname [join $parts]
    }
    return $fullname
}

proc Layers::NameSplit { name } {
    return [GidUtils::Split $name //]
}

proc Layers::GetAutomaticLayerName { {prefix ""} } {
    set LayerNames [GiD_Layers list]
    set name ""
    for {set i 0} {$i<10000} {incr i} {
        set fullname [Layers::NameJoin $prefix Layer${i}]
        if { [lsearch -exact $LayerNames $fullname] == -1 } {
            set name $fullname
            break
        }
    }
    return $name
}

proc Layers::CreateNewLayer { {T ""} {name ""} } {
    set prefix ""
    if { $name == "" } {
        set name [Layers::GetAutomaticLayerName $prefix]
    } else {
        if { [GiD_Layers is_forbidden_name  $name] } {
            WarnWin [_ "Bad layer name"]
            return ""
        }
    }
    GiD_Process 'Layers New $name escape
    update ;#to fill in again the treectrl to create the item
    if { $T != "" } {
        set item [$T item id "tag [Layers::EncodeName $name]"]
        if { $item != "" } {
            $T activate $item
            Layers::BeginEditLayers $T
        }
    }
    return $name
}

proc Layers::ClickTableList { x y T } {
    #     focus $T

    set info [$T identify $x $y]
    if { [lindex $info 0] == "item" && [llength $info] >= 4 } {
        set item [lindex $info 1]
        set col [lindex $info 3]
    } elseif { [lindex $info 0] == "header" && [lindex $info 1] == "0" } {
        if { [$T column cget C0 -arrow] == "up" } {
            $T column configure C0 -arrow down
            $T item sort 0 -dictionary -increasing
        } else {
            $T column configure C0 -arrow up
            $T item sort 0 -dictionary -decreasing
        }
        return ""
    } else {
        return ""
    }

    if { $col != 0 } {
        if { ![$T selection includes $item] } {
            $T selection clear
            $T selection add $item
        }
    }
    if { $col == 0 } {
        #Layers::SetLayersTo TOUSE $T
    } elseif { $col == 1 } {
        Layers::SetLayersTo TOUSE $T
    } elseif { $col == 2 } {
        Layers::SetLayersToColor $T
    } elseif { $col == 3 } {
        if { [$T item element cget $item C$col elemImgAny -image] == [gid_themes::GetImage layer_off.png small_icons] } {
            Layers::SetLayersTo ON $T
        } else {
            Layers::SetLayersTo OFF $T
        }
    } elseif { $col == 4 } {
        if { [$T item element cget $item C$col elemImgAny -image] == [gid_themes::GetImage lock_on.png small_icons] } {
            Layers::SetLayersTo UNFREEZE $T
        } else {
            Layers::SetLayersTo FREEZE $T
        }
    } elseif { $col == 5 } {
        if { [$T item element cget $item C$col elemImgAny -image] == [gid_themes::GetImage transparent_off.png small_icons] } {
            Layers::SetLayersTo TRANSPARENT $T
        } else {
            Layers::SetLayersTo OPAQUE $T
        }
    } elseif { $col == 6 } {
        if { [$T item element cget $item C$col elemImgAny -image] == [gid_themes::GetImage bring_to_front.png small_icons] } {
            Layers::SetLayersTo BRINGTOFRONT $T
        } else {
        }
    }

    if { $col != 0 } {
        return -code break
    }
    return ""
}

proc Layers::InvertSelectionTableList { T } {
    set newselection {}
    foreach item [$T item range 0 end] {
        if { ![$T selection includes $item] } {
            lappend newselection $item
        }
    }
    $T selection clear
    foreach item $newselection {
        $T selection add $item
    }
}

proc Layers::DoubleClickTableList { x y T } {
    set info [$T identify $x $y]
    if { [lindex $info 0] == "item" && [llength $info] >= 4 } {
        set item [lindex $info 1]
        set col [lindex $info 3]
    } else {
        return
    }

    if { $col == 0 } {
        $T selection clear
        $T selection add $item
        Layers::SetLayersTo TOUSE $T
    }
}

proc Layers::ListLayers { {state normal} } {
    set layers {}
    foreach layer [GiD_Layers list] {
        if { [GiD_Layers get state $layer] == $state } {
            lappend layers $layer
        }
    }
    return [lsort -unique -dictionary $layers]
}

proc Layers::ListChildLayers { parent {state normal} } {
    set layers {}
    foreach layer [GiD_Layers list $parent] {
        set fullname [Layers::NameJoin $parent $layer]
        if { [GiD_Layers get state $fullname] == $state } {
            lappend layers $layer
        }
    }
    return [lsort -unique -dictionary $layers]
}

proc Layers::IsItemFolderOfLayers { T item } {
    return [$T item numchildren $item]
}

proc Layers::FillTableInfoLayers { T {parent ""} {parentitem root} } {
    variable _tree_isopen
    if { [winfo exists $T] } {
        array unset selectionitems
        #try to mantain the previous state: open, selected
        foreach tree_id [$T item range 0 end] {
            set fullnameencoded [$T item tag names $tree_id]
            set selectionitems($fullnameencoded) [$T selection includes $tree_id]
        }
        $T item delete all ;#coult try not to delete and rebuild all...
        Layers::FillTableInfoLayers_New $T $parent $parentitem
        foreach tree_id [$T item range 0 end] {
            set fullnameencoded [$T item tag names $tree_id]
            set fullname [Layers::DecodeName $fullnameencoded]
            if { [info exists _tree_isopen($fullname)] } {
                $T item expand $tree_id
            } else {
                $T item collapse $tree_id
            }
            if { [info exists selectionitems($fullnameencoded)] && $selectionitems($fullnameencoded) } {
                $T selection add $tree_id
            }
        }
        set id [$T item id "tag [Layers::EncodeName [GiD_Info Project LayerToUse]]"]
        if { $id != "" } {
            $T activate $id
        }
        if { [$T selection count] == 1 } {
            $T see [$T selection get]
        }
        set ::TreeCtrl::Priv(DirCnt,$T) [$T item count] ;#to use drag file-list bindings
    }
}

proc Layers::IfNothingSelectedSelectAll { T } {
    set sel [$T selection get]
    if { $sel == "" } {
        $T selection add all
        set sel [$T selection get]
    }
    return $sel
}

proc Layers::GetSelectedLayernames { T } {
    if { ![winfo exists $T] } { return }
    set layer_names ""
    set sel [$T selection get]
    foreach item $sel {
        lappend layer_names [Layers::DecodeName [$T item tag names $item]]
    }
    set layer_names [lsort -unique -dictionary $layer_names]
    return $layer_names
}

proc Layers::FillTableInfoLayers_New { T {parent ""} {parentitem root} } {

    #     if { ![winfo exists $T] } { return }

    set LayerNames [Layers::ListChildLayers $parent]
    set LayerToUse [GiD_Info Project LayerToUse]
    foreach i $LayerNames {
        set fullname [Layers::NameJoin $parent $i]
        set has_childs [llength [Layers::ListChildLayers $fullname]]
        if { $has_childs } {
            set isbutton yes
        } else {
            set isbutton no
        }

        set item [$T item create -button $isbutton -tags [Layers::EncodeName $fullname]]
        $T item lastchild $parentitem $item
        $T item style set $item C0 styAnyRead
        $T item element configure $item C0 elemTxtRead -text $i

        $T item style set $item C1 styAnyImage
        if { $fullname == $LayerToUse } {
            #                 $T item element configure $item C0 elemTxtRead -text $i + elemImgAny -image [gid_themes::GetImage in_use.png small_icons]
            $T item element configure $item C1 elemImgAny -image [gid_themes::GetImage in_use.png small_icons]
            #                 $T item state set $item touse
        } else {
            #                 $T item element configure $item C0 elemTxtRead -text $i
            $T item element configure $item C1 elemImgAny -image [gid_themes::GetImage blank.png small_icons];#void image
            #                 $T item state set $item !touse
        }
        set cur_col [string range [GiD_Layers get color $fullname] 0 6] ;#avoid last two characters of alpha
        $T item style set $item C2 styRectColor
        $T item element configure $item C2 elemRectColor -fill $cur_col

        $T item style set $item C3 styAnyImage
        if { [GiD_Layers get visible $fullname] } {
            $T item element configure $item C3 elemImgAny -image [gid_themes::GetImage layer_on.png small_icons]
            #                 $T item state set $item on
        } else {
            $T item element configure $item C3 elemImgAny -image [gid_themes::GetImage layer_off.png small_icons]
            #                 $T item state set $item !on
        }


        $T item style set $item C4 styAnyImage
        if { [GiD_Layers get frozen $fullname] } {
            $T item element configure $item C4 elemImgAny -image [gid_themes::GetImage lock_on.png small_icons]
            #                 $T item state set $item freeze
        } else {
            $T item element configure $item C4 elemImgAny -image [gid_themes::GetImage lock_off.png small_icons]
            #                 $T item state set $item !freeze
        }
        $T item style set $item C5 styAnyImage
        if { [GiD_Layers get opaque $fullname] } {
            $T item element configure $item C5 elemImgAny -image [gid_themes::GetImage transparent_off.png small_icons]
            #                 $T item state set $item transparent
        } else {
            $T item element configure $item C5 elemImgAny -image [gid_themes::GetImage transparent_on.png small_icons]
            #                 $T item state set $item !transparent
        }

        $T item style set $item C6 styAnyImage
        if { [GiD_Layers get back $fullname] } {
            $T item element configure $item C6 elemImgAny -image [gid_themes::GetImage bring_to_front.png small_icons]
            #                 $T item state set $item back
        } else {
            $T item element configure $item C6 elemImgAny -image [gid_themes::GetImage blank.png small_icons] ;#void image
            #                 $T item state set $item !back
        }


        if { $has_childs } {
            Layers::FillTableInfoLayers_New $T $fullname $item
        }
    }
    if { [$T column cget C0 -arrow] == "up" } {
        $T item sort 0 -dictionary -decreasing
    } else {
        $T item sort 0 -dictionary -increasing
    }
}

proc Layers::ChangeParentLayers { T dragged_list new_parent_name } {
    set cmd ""
    foreach item $dragged_list {
        set fullname [Layers::DecodeName [$T item tag names $item]]
        lappend cmd ChangeParent $fullname $new_parent_name
    }
    return $cmd
}

proc Layers::ReceiveDragLayers { state T dragged_list dst } {
    if { $state == "receive" } {
        set new_parent_name [Layers::DecodeName [$T item tag names $dst]]
        set cmd [Layers::ChangeParentLayers $T $dragged_list $new_parent_name]
        GiD_Process 'Layers {*}$cmd escape
    } elseif { $state == "end" } {
        if { ![info exists ::ReceiveDragLayers_prev_state] } {
            return ""
        }
        if { $::ReceiveDragLayers_prev_state != "receive" } {
            #trick, if previous to "end" was not "receive" then was dropped
            #outside any parent, then set as a first level layer without parent
            set dragged_list [$T selection get]
            set new_parent_name ""
            set cmd [Layers::ChangeParentLayers $T $dragged_list $new_parent_name]
            GiD_Process 'Layers {*}$cmd escape
        }
    }
    set ::ReceiveDragLayers_prev_state $state
}

proc Layers::SetLayersToRename { T item newname } {
    if { $newname == ""} return
    if { $item == 0 } {
        WarnWin [_ "Root folder can't be edited"]
        return
    }
    set fullname [Layers::DecodeName [$T item tag names $item]]
    set fullparentname [GiD_Layers get parent $fullname]
    set newfullname [Layers::NameJoin $fullparentname $newname]
    if { $fullname != $newfullname } {
        GiD_Process 'Layers ChangeName $fullname $newfullname escape
    }
}

proc Layers::BeginEditLayers { T } {
    set I [$T item id active]
    set C 0
    set E elemTxtRead
    ::TreeCtrl::FileListEdit $T $I $C $E
}

proc Layers::LayersCreateListWin { w } {
    package require treectrl

    # Get default colors
    set w_tmp [listbox .listbox]
    set SystemHighlight [$w_tmp cget -selectbackground]
    set SystemHighlightText [$w_tmp cget -selectforeground]
    destroy $w_tmp
    if { $::tcl_platform(platform) == "unix" } {
        ## # I hate that gray selection color
        ## set SystemHighlight #316ac5
        # ::GidPriv(Color,BotonActivo) = #5ac9da
        # set SystemHighlightText White
        set SystemHighlight $::GidPriv(Color,BotonActivo)
        set SystemHighlightText Black
    }


    set vsb $w.vsb1
    set hsb $w.hsb1
    #pseudottk::
    set T [treectrl $w.t -xscrollcommand [list $hsb set] -yscrollcommand [list $vsb set]]
    ttk::scrollbar $vsb -orient vertical   -command [list $T yview]
    ttk::scrollbar $hsb -orient horizontal -command [list $T xview]

    set height [font metrics [$T cget -font] -linespace]
    if {$height < 18} {
        set height 18
    }

    $T configure -indent 15 -itemheight $height -selectmode extended \
        -showroot 0 -showrootbutton 0 -showbuttons 1 -showlines 1 \
        -highlightthickness 0 -borderwidth 0 \
        -xscrollincrement 20 -yscrollincrement 20


    #------------------test cabeceras treectrl------------
    gid_themes::ConfigureHeaders $T
    #-----------------part 1 fin----------------------

    set coltxt [list [_ "Name"] { } [_ "C#C#Color"] [_ "I/O#C#On/Off"] \
        [_ "F/U#C#Freeze/Unfreeze"] [_ "Tr#C#transparent"] [_ "B#C#back"]]

    #     $T state define touse
    #     $T state define on
    #     $T state define freeze
    #     $T state define transparent
    #     $T state define back

    set i 0
    foreach txt $coltxt {
        $T column create -text $txt -tags C$i -weight 0
        #$T column create -text $txt -tags C$i -weight 0 -justify center \
        #    -gridrightcolor gray90 -itembackground {linen {} #00ff00 {}}
        incr i
    }
    $T column configure C0 -weight 1 -arrow down
    #     $T column configure tail -width 0
    #     $T column configure tail -visible 0
    $T configure -treecolumn C0

    #------------------part 2 test cabeceras treectrl------------
    gid_themes::ApplyHeadersConfig $T
    #-----------------fin test----------------------


    # Create elements
    $T element create elemImgAny image
    $T element create elemTxtRead text -fill [list $SystemHighlightText {selected focus}] \
        -lines 1
    $T element create elemRectSel rect -fill [list $SystemHighlight {selected focus} gray {selected !focus}] \
        -showfocus yes
    $T element create elemRectColor rect -width 30
    # Create styles using the elements
    set S [$T style create styAnyRead]
    $T style elements $S {elemRectSel elemTxtRead elemImgAny}
    $T style layout $S elemImgAny -expand ns
    $T style layout $S elemTxtRead -padx 4 -expand ns -squeeze x
    $T style layout $S elemRectSel -union [list elemTxtRead] -iexpand ns -ipadx 2

    set S [$T style create styRectColor]
    $T style elements $S {elemRectColor elemTxtRead}
    $T style layout $S elemRectColor -union [list elemTxtRead] -iexpand ns -ipadx 2

    set S [$T style create styAnyImage]
    $T style elements $S {elemImgAny}
    $T style layout $S elemImgAny -expand ns


    set item root
    $T item configure $item -button yes
    $T item style set $item C0 styAnyRead
    $T item element configure $item C0 elemTxtRead -text [_ "Root"]
    #$T item element configure $item C0 elemImgAny -image [gid_themes::GetImage layer.png small_icons]


    # List of lists: {column style element ...} specifying text elements
    # the user can edit
    TreeCtrl::SetEditable $T {
        {C0 styAnyRead elemTxtRead}
    }

    # List of lists: {column style element ...} specifying elements
    # the user can click on or select with the selection rectangle
    TreeCtrl::SetSensitive $T {
        {C0 styAnyRead elemTxtRead}
    }


    # List of lists: {column style element ...} specifying elements
    # added to the drag image when dragging selected items
    TreeCtrl::SetDragImage $T {
        {C0 styAnyRead elemTxtRead}
    }

    #$T notify install <Drag-begin>
    $T notify install <Drag-end>
    $T notify install <Drag-receive>
    #     $T notify install <Edit-begin>
    #     $T notify install <Edit-end>
    $T notify install <Edit-accept>

    #$T notify bind DragTag <Drag-begin> { Layers::ReceiveDragLayers begin %T %l %I}
    $T notify bind DragTag <Drag-end> { Layers::ReceiveDragLayers end %T %l %I}
    $T notify bind DragTag <Drag-receive> { Layers::ReceiveDragLayers receive %T %l %I}

    $T notify bind EditTag <Edit-accept> { Layers::SetLayersToRename %T %I %t }

    bind $T <Button-1> [list Layers::ClickTableList %x %y $T]
    bind $T <Double-Button-1> [list Layers::DoubleClickTableList %x %y $T]
    #bind $T <Return> [list Layers::SetLayersTo TOUSE $T]
    bind $T <Return> [list Layers::CreateNewLayer $T]
    bind $T <Key-Delete> [list Layers::SetLayersToDelete $T]
    bind $T <Alt_L> [list Layers::InvertSelectionTableList $T]
    bind $T <Alt_R> [list Layers::InvertSelectionTableList $T]
    bind $T <Meta_L> [list Layers::InvertSelectionTableList $T]
    bind $T <Meta_R> [list Layers::InvertSelectionTableList $T]
    bind $T <F2> [list Layers::BeginEditLayers $T]

    bind $T <Button-$::gid_right_button> "[list Layers::MenuContextualLayer %W %x %y] ; break"

    grid $T $vsb -sticky nsew
    grid configure $vsb -sticky ns
    grid $hsb -sticky ew
    grid remove $vsb $hsb
    bind $T <Configure> [list ConfigureListScrollbars $T $hsb $vsb]

    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1

    #     bindtags $T [list $T DemoExplorer TreeCtrlFileList TreeCtrl [winfo toplevel $T] all]
    bindtags $T [list $T TreeCtrlFileList TreeCtrl [winfo toplevel $T] all]

    #to avoid mousewheel of layers TreeCtrl propagated to gid_togl widget

    #bind $T <MouseWheel> "[bind [winfo class $T] <MouseWheel>] ; break"
    #fulltktree is removing the binding of TreeCtrl agressively with bind TreeCtrl <MouseWheel> {} !!!
    #then can't ask here its binding, use a copy of the original code!!
    bind $T <MouseWheel> {%W yview scroll [expr {- (%D / 120) * 4}] units ; break}

    set column_items {name touse color visibility state transparency back}
    TreeSelectColumnsVisibility $T Layers $column_items

    $T notify bind $T <Expand-after> {Layers::TreeExpandAfter %T %I}
    $T notify bind $T <Collapse-after> {Layers::TreeCollapseAfter %T %I}
    return $T
}

proc Layers::TreeExpandAfter { T item } {
    variable _tree_isopen
    set fullname [Layers::DecodeName [$T item tag names $item]]
    set _tree_isopen($fullname) 1
}

proc Layers::TreeCollapseAfter { T item } {
    variable _tree_isopen
    set fullname [Layers::DecodeName [$T item tag names $item]]
    unset -nocomplain _tree_isopen($fullname)
}

proc Layers::ResetTreeStatus { } {
    variable _tree_isopen
    array unset _tree_isopen
}

proc Layers::ChangeLayers { {w .gid.central.wlay} } {
    global busy
    if { [info exists busy] && $busy } {
        #avoid Tcl error when pressing continuously a key that invoke a procedure
        return
    }
    set busy 1
    if { ![info exists ::GidPriv(LayerLayout)] } {
        set ::GidPriv(LayerLayout) RIGHT
        #do not use INSIDERIGHT because sharing gid.ini cause problems for previous GiD's in ReadDefaultValues
    }
    if { $::GidPriv(LayerLayout) == "RIGHT" } {
        set inside 1
    } else {
        set inside 0
    }
    #problems with panedwindow: without after idle have bad size when restarting
    #gid with Layers window inside
    if { [catch { Layers::ChangeLayers_New $w $inside } err options] } {
        set busy 0
        return -options $options $err
    }
    set busy 0
}


proc Layers::WriteLayerGeomToVar { w what geomname {InitComm ""}} {
    global GidPriv
    if { ![info exists ::GidPriv(LayerLayout)] } {
        set ::GidPriv(LayerLayout) RIGHT
        #do not use INSIDERIGHT because sharing gid.ini cause problems for previous GiD's in ReadDefaultValues
    }
    set trans 1
    update idletasks
    if { $::GidPriv(LayerLayout) == "RIGHT" } {
        #maintain all dimensions except the new width
        if { [info exists GidPriv($geomname)] } {
            set prevgeom [lindex $GidPriv($geomname) 1]
            lassign [split $prevgeom x+] width height x y
            set width [winfo width $w]
        } else {
            set width [winfo width $w]
            set height [winfo height $w]
            set x [winfo x $w]
            set y [winfo y $w]
        }
    } else {
        #"OUTSIDE"
        set width [winfo width $w]
        set height [winfo height $w]
        set x [winfo x $w]
        set y [winfo y $w]
    }
    set geom ${width}x${height}+${x}+${y}
    set GidPriv($geomname) [list $what $geom $trans $InitComm]
}

########################
#   Some changes to take account GidPriv(pwCentral), a panedwindow to manage several windows
########################
proc Layers::ChangeLayers_New { w {inside 1} } {
    global GidPriv

    if { $inside } {
        Layers::OpenWindowInside $w
        if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
        if { [winfo class $w] == "Toplevel" } return

        #######
        # pwCentral is a panedwindow to manage $grWindow with other windows in GiD
        if {![info exists GidPriv(pwCentral)]} return
        if {![winfo exists $GidPriv(pwCentral)]} return
        #######


        ttk::frame $w.caption -borderwidth 1 -style sunken.TFrame
        GidHelp $w.caption [_ "Double-click toggle window or inline visualization"]
        ttk::label $w.caption.title -state disabled -anchor center -text [_ "Double click here to tear off the window"]
        tk::button $w.caption.close -image [gid_themes::GetImage close17.png] \
            -command {GidUtils::CloseWindow LAYER} -borderwidth 0 -relief flat

        grid $w.caption.title $w.caption.close -sticky ew
        grid configure $w.caption.close -sticky w
        grid columnconfigure $w.caption 0 -weight 1
        bind $w.caption.title <Double-Button-1> [list Layers::SwitchWindowInsideOutside]

        ttk::frame $w.body
        grid $w.caption -sticky ew
        grid $w.body -sticky nsew
        grid rowconfigure $w 1 -weight 1
        grid columnconfigure $w 0 -weight 1

        if { $::GidPriv(ShowGroupsTab) } {
            grid rowconfigure $w.body 0 -weight 1
            grid columnconfigure $w.body 0 -weight 1
            set nb [ttk::notebook $w.body.nb]
            ttk::notebook::enableTraversal $nb
            set f_layers [ttk::frame $nb.f_layers]
            $nb add $f_layers -text [_ "Layers"] -underline 0
            Layers::ChangeLayers_New2 $f_layers
            set f_groups [ttk::frame $nb.f_groups]
            $nb add $f_groups -text [_ "Groups"] -underline 0
            Groups::Win $f_groups
            grid $nb -sticky nsew
            if { [info exists ::GidPriv(LayersOrGroupsCurrentTab)] } {
                $nb select $::GidPriv(LayersOrGroupsCurrentTab)
            }
        } else {
            Layers::ChangeLayers_New2 $w.body
        }

        update idletasks
        if { [info exists ::GidPriv(PreLayersWindowGeom)] } {
            set w1 [lindex [split [lindex $::GidPriv(PreLayersWindowGeom) 1] x] 0]
        } else {
            set w1 [winfo reqwidth $w.body]
        }
        set sepw 4
        set w0 [expr [winfo width $GidPriv(pwCentral)]-$w1-$sepw]
        if { $w0<=0 || $w1<=0 } {
            set w0 750
            set w1 250
        }

        set wsash [expr [lsearch [$GidPriv(pwCentral) panes] $w]-1]
        if {$wsash < 0} {
            set wsash 0
        }

        if { [winfo class $GidPriv(pwCentral)] == "Panedwindow" } {
            #problems with panedwindow: without update idletasks have bad size when #reopening again the layers window inside
            update idletasks
            $GidPriv(pwCentral) sash place $wsash $w0 0 ;#raise reshape_cb of gid_togl
        } elseif { [winfo class $GidPriv(pwCentral)] == "TPanedwindow" } {
            #ttk::panedwindow
            $GidPriv(pwCentral) sashpos $wsash $w0
        }

        set ::GidPriv(LayerLayout) "RIGHT"
        bind $w <Configure> [list Layers::WriteLayerGeomToVar $w OPEN PreLayersWindowGeom Layers::ChangeLayers]
        # internal window --> no finishbutton
        set ::wChangeLayFrame $w.body
        set ::wChangeLayButtonFrame ""
    } else {
        Layers::OpenWindowOutside $w
        if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
        set ::GidPriv(LayerLayout) "OUTSIDE"
        wm protocol $w WM_DELETE_WINDOW [list GidUtils::CloseWindow LAYER]

        ttk::frame $w.caption -borderwidth 1 -style sunken.TFrame
        GidHelp $w.caption [_ "Double-click toggle window or inline visualization"]

        ttk::label $w.caption.title -state disabled -anchor center -text [_ "Double click here to integrate the window"]

        grid $w.caption.title -sticky ew
        grid columnconfigure $w.caption 0 -weight 1
        bind $w.caption.title <Double-Button-1> [list Layers::SwitchWindowInsideOutside]

        ttk::frame $w.body
        grid $w.caption -sticky ew
        grid $w.body -sticky nsew
        grid rowconfigure $w 1 -weight 1
        grid columnconfigure $w 0 -weight 1

        if { $::GidPriv(ShowGroupsTab) } {
            grid rowconfigure $w.body 0 -weight 1
            grid columnconfigure $w.body 0 -weight 1
            set nb [ttk::notebook $w.body.nb]
            ttk::notebook::enableTraversal $nb
            set f_layers [ttk::frame $nb.f_layers]
            $nb add $f_layers -text [_ "Layers"] -underline 0
            Layers::ChangeLayers_New2 $f_layers
            set f_groups [ttk::frame $nb.f_groups]
            $nb add $f_groups -text [_ "Groups"] -underline 0
            Groups::Win $f_groups
            grid $nb -sticky nsew
            if { [info exists ::GidPriv(LayersOrGroupsCurrentTab)] } {
                $nb select $::GidPriv(LayersOrGroupsCurrentTab)
            }
        } else {
            Layers::ChangeLayers_New2 $w.body
        }

        #add lower buttons
        ttk::frame $w.buts -style BottomFrame.TFrame
        ttk::button $w.buts.can -text [_ "Close"] -command [list GidUtils::CloseWindow LAYER] -style BottomFrame.TButton
        grid $w.buts.can -sticky ew -padx 5 -pady 6
        grid $w.buts -sticky ews -pady {2 0}
        grid anchor $w.buts center
        set ::wChangeLayFrame $w.body
        set ::wChangeLayButtonFrame $w.buts
    }
}

proc Layers::ChangeLayers_New2 { w } {
    global wChangeLay GidPriv
    set wChangeLay $w
    ttk::frame $w.middle
    set T [Layers::LayersCreateListWin $w.middle]
    set GidPriv(Layers,table) $T

    ttk::frame $w.top -style ForcedFrame
    #new.png small_icons
    ttk::button $w.top.newlayer -image [gid_themes::GetImage layers_new.png small_icons]  -command [list Layers::CreateNewLayer $T] -style IconButton
    GidHelp $w.top.newlayer [_ "Create a new layer at root level."]
    ttk::button $w.top.newfolder -image [gid_themes::GetImage newfolder.png small_icons]  -command [list Layers::CreateNewLayerFolder $T] -style IconButton
    GidHelp $w.top.newfolder [_ "Create a child layer with the selected parent."]
    #ttk::button $w.top.rename -image [gid_themes::GetImage rename.png small_icons]  -command [list Layers::BeginEditLayers $T]
    #GidHelp $w.top.rename [_ "Rename a layer."]
    ttk::button $w.top.delete -image [gid_themes::GetImage delete.png small_icons]  -command [list Layers::SetLayersToDelete $T] -style IconButton
    GidHelp $w.top.delete [_ "Delete a layer."]
    ttk::menubutton $w.top.sendto -image [gid_themes::GetImage sendto.png small_icons] -menu $w.top.sendto.m \
        -takefocus 1 -style IconButton
    GidHelp $w.top.sendto \
        [_ "Send the selected entities to a layer.\
        If 'Also lower entities' is checked, the subentities are also send to the layer"]
    Layers::CreateMenuSendTo $w.top.sendto.m $T
    ttk::menubutton $w.top.toback -image [gid_themes::GetImage send_to_back.png small_icons] -text [_ "To back"] -menu $w.top.toback.m \
        -takefocus 1 -style IconButton
    GidHelp $w.top.toback \
        [_ "Send entities to back/front of a layer (entities in back are not drawn and frozen)"]
    Layers::CreateMenuBack $w.top.toback.m $T

    ttk::menubutton $w.top.entitylayer -menu $w.top.entitylayer.m -width 0 -image [gid_themes::GetImage layerofentity.png small_icons] -style IconButton
    GidHelp $w.top.entitylayer [_ "This option permmits to select one entity and set its layer as layer to use."]
    menu $w.top.entitylayer.m -postcommand [list Layers::LayerToUseEntities $w.top.entitylayer.m]

    ttk::button $w.top.filter -image [gid_themes::GetImage filter.png small_icons]  -command [list Layers::SelectLayersByRegexp $T] -style IconButton
    GidHelp $w.top.filter \
        [_ "Select layers with filters (for example: 'b*' select all the layers with name beginning by 'b')"]

    ttk::button $w.top.list_entities -image [gid_themes::GetImage list.png small_icons] \
        -command [list Layers::ListEntitiesSelection $T] -style IconButton
    GidHelp $w.top.list_entities [_ "List layer entities"]

    grid $w.top.newlayer $w.top.newfolder $w.top.delete \
        $w.top.sendto $w.top.toback $w.top.entitylayer $w.top.filter \
        $w.top.list_entities -sticky w

    grid $w.top -sticky ew
    grid $w.middle -sticky wens

    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 1 -weight 1

    Layers::FillInfo
    focus $T
}

proc Layers::CreateMenuBack { m T } {
    menu $m -postcommand [list Layers::SendLayerEntitiesToBack $m $T] -activeborderwidth 1
}

proc Layers::CreateMenuSendTo { m T } {
    menu $m -postcommand [list Layers::SendLayerEntities $m $T] -activeborderwidth 1
}

proc Layers::MenuContextualLayer { T x y } {
    set category Layers
    set w $T.menucontextuallayer
    if { [winfo exists $w] } {
        destroy $w
    }

    menu $w

    if { [$T selection count] == 0 } {
        # try select what's under the mouse
        set info [ $T identify $x $y]
        if { [lindex $info 0] == "item" && [llength $info] >= 4 } {
            set item [lindex $info 1]
            set col [lindex $info 3]
            # set name [ $tbl item element cget $item C0 elemTxtRead -text]
            set name [Layers::DecodeName [ $T item tag names $item]]
            set LayersSelection [ list $name]
            # $T select $item $col
            $T selection clear
            $T selection add $item
        }
    }

    $w add command -label [_ "New layer#C#layer"] -command [list Layers::CreateNewLayer $T] \
        -image [gid_themes::GetImage layers_new.png small_icons] -compound left
    $w add command -label [_ "New child#C#layer"] -command [list Layers::CreateNewLayerFolder $T] \
        -image [gid_themes::GetImage newfolder.png small_icons] -compound left

    set nfolder 0
    set nsingle 0
    foreach item [$T item range 0 end] {
        if { [$T selection includes $item] } {
            if { [Layers::IsItemFolderOfLayers $T $item] } {
                incr nfolder
            } else {
                incr nsingle
            }
        }
    }
    if { [expr {$nsingle+$nfolder}] == 1 } {
        $w add command -label [_ "Rename"] -command [list Layers::BeginEditLayers $T] -state normal \
            -image [gid_themes::GetImage rename.png small_icons] -compound left
    } else {
        $w add command -label [_ "Rename"] -state disabled \
            -image [gid_themes::GetImage rename.png small_icons] -compound left
    }
    if { [expr {$nsingle+$nfolder}] > 0 } {
        $w add command -label [_ "Delete"] -command [list Layers::SetLayersToDelete $T] -state normal \
            -image [gid_themes::GetImage delete.png small_icons] -compound left
    } else {
        $w add command -label [_ "Delete"] -state disabled \
            -image [gid_themes::GetImage delete.png small_icons] -compound left
    }
    if { [expr {$nsingle+$nfolder}] == 1 } {
        $w add command -label [_ "To use"] -command [list Layers::SetLayersTo TOUSE $T] -state normal \
            -image [gid_themes::GetImage in_use.png small_icons] -compound left
    }
    $w add separator

    if { [expr {$nsingle+$nfolder}] > 0 } {
        set state normal
    } else {
        set state disabled
    }
    set item color
    if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] || $::GidPriv(ColumnVisible,$category,$item) } {
        $w add command -label [_ "Color"]... -command [list Layers::SetLayersToColor $T] -state $state \
            -image [gid_themes::GetImage color.png small_icons] -compound left
    }
    if { [expr {$nsingle+$nfolder}] ==1 } {
        set item [$T selection get 0]
        set fullname [Layers::DecodeName [$T item tag names $item]]
        set item visibility
        if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] || $::GidPriv(ColumnVisible,$category,$item) } {
            if { [GiD_Layers get visible $fullname] } {
                $w add command -label [_ "Off#C#verb"] -command [list Layers::SetLayersTo OFF $T] -state $state \
                    -image [gid_themes::GetImage layer_off.png small_icons] -compound left
            } else {
                $w add command -label [_ "On#C#verb"] -command [list Layers::SetLayersTo ON $T] -state $state \
                    -image [gid_themes::GetImage layer_on.png small_icons] -compound left
            }
        }
        set item state
        if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] || $::GidPriv(ColumnVisible,$category,$item) } {
            if { [GiD_Layers get frozen $fullname] } {
                $w add command -label [_ "Unfreeze#C#verb"] -command [list Layers::SetLayersTo UNFREEZE $T] -state $state \
                    -image [gid_themes::GetImage lock_off.png small_icons] -compound left
            } else {
                $w add command -label [_ "Freeze#C#verb"] -command [list Layers::SetLayersTo FREEZE $T] -state $state \
                    -image [gid_themes::GetImage lock_on.png small_icons] -compound left
            }
        }
        set item transparency
        if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] || $::GidPriv(ColumnVisible,$category,$item) } {
            if { [GiD_Layers get opaque $fullname] } {
                $w add command -label [_ "Transparent#C#verb"] -command [list Layers::SetLayersTo TRANSPARENT $T] -state $state \
                    -image [gid_themes::GetImage transparent_on.png small_icons] -compound left
            } else {
                $w add command -label [_ "Opaque#C#verb"] -command [list Layers::SetLayersTo OPAQUE $T] -state $state \
                    -image [gid_themes::GetImage transparent_off.png small_icons] -compound left
            }
        }
    } else {
        set item visibility
        if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] || $::GidPriv(ColumnVisible,$category,$item) } {
            $w add command -label [_ "On#C#verb"] -command [list Layers::SetLayersTo ON $T] -state $state \
                -image [gid_themes::GetImage layer_on.png small_icons] -compound left
            $w add command -label [_ "Off#C#verb"] -command [list Layers::SetLayersTo OFF $T] -state $state \
                -image [gid_themes::GetImage layer_off.png small_icons] -compound left
        }
        set item state
        if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] || $::GidPriv(ColumnVisible,$category,$item) } {
            $w add command -label [_ "Freeze#C#verb"] -command [list Layers::SetLayersTo FREEZE $T] -state $state \
                -image [gid_themes::GetImage lock_on.png small_icons] -compound left
            $w add command -label [_ "Unfreeze#C#verb"] -command [list Layers::SetLayersTo UNFREEZE $T] -state $state \
                -image [gid_themes::GetImage lock_off.png small_icons] -compound left
        }
        set item transparency
        if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] || $::GidPriv(ColumnVisible,$category,$item) } {
            $w add command -label [_ "Opaque#C#verb"] -command [list Layers::SetLayersTo OPAQUE $T] -state $state \
                -image [gid_themes::GetImage transparent_off.png small_icons] -compound left
            $w add command -label [_ "Transparent#C#verb"] -command [list Layers::SetLayersTo TRANSPARENT $T] -state $state \
                -image [gid_themes::GetImage transparent_on.png small_icons] -compound left
        }
    }
    $w add separator
    $w add command -label [_ "Select all"] -command [list $T selection add all]
    $w add separator
    if { $nsingle == 1 && $nfolder == 0 } {
        set state normal
    } else {
        #set state disabled
        set state normal ;#allow for example Send to->Self-dependencies
    }
    $w add cascade -label [_ "Send to"] -menu $w.sendto -state $state -image [gid_themes::GetImage sendto.png small_icons] -compound left
    Layers::CreateMenuSendTo $w.sendto $T
    $w add cascade -label [_ "To back"] -menu $w.back \
        -image [gid_themes::GetImage send_to_back.png small_icons] -compound left
    Layers::CreateMenuBack $w.back $T
    $w add command -label [_ "List layer entities"] -command [list Layers::ListEntitiesSelection $T] -state normal \
        -image [gid_themes::GetImage list.png small_icons] -compound left
    $w add separator
    $w add command -label [_ "Filter"] -command [list Layers::SelectLayersByRegexp $T] \
        -image [gid_themes::GetImage filter.png small_icons] -compound left
    if { $::GidPriv(ShowGroupsTab) } {
        $w add separator
        if { [expr {$nsingle+$nfolder}] > 0 } {
            set state normal
        } else {
            set state disabled
        }
        $w add command -label [_ "Layers to groups"] -command [list Layers::LayersToGroupsAsk $T] -state $state \
            -image [gid_themes::GetImage layers.png small_icons] -compound left
    }
    $w add command -label [_ "Select columns"]... -command [list Layers::SelectColumnsWindow]

    set x [expr [winfo rootx $T]+$x+2]
    set y [expr [winfo rooty $T]+$y]
    GiD_PopupMenu $w $x $y
}

proc Layers::LayersToGroupsAsk { T } {
    set layer_names [Layers::GetSelectedLayernames $T]
    if { [llength $layer_names] } {
        set retval [MessageBoxOptionsButtons [_ "Layers to groups"] \
            [_ "Are you sure to create groups from the selected layers?"] \
            {0 1} [list [_ "Yes"] [_ "No#C#I don't want to do that"]] question ""]
        if { $retval == 0 } {
            #$::CUSTOM_LIB_USE_NATIVE_GROUPS ??
            #if { $::GidPriv(ShowGroupsTab) } ??
            if { $::CUSTOM_LIB_USE_NATIVE_GROUPS } {
                Groups::LayersToGroups $layer_names
            } else {
                #package require customLib
                if { [GroupsXmlDocExists] } {
                    gid_groups_conds::layers_to_groups_do $layer_names
                }
            }
        }
    }
}

proc Layers::SelectColumnsWindow { } {
    variable SelectColumn ;#local temporary value
    set category Layers
    set tree $::GidPriv(Layers,table)
    set column_items {name touse color visibility state transparency back}
    set column_labels [list [_ "Name"] [_ "To use"] [_ "Color"] [_ "Visibility"] [_ "State"] [_ "Transparency"] [_ "Back"]]
    set w .gid.selectcolumns
    InitWindow2 $w -title [_ "Select %s columns" [_ $category]] \
        -geometryvariable PreSelect${category}ColumnsWindowGeom \
        -initcommand Select${category}ColumnsWindow -ontop
    set c [CreateScrolledCanvas $w.body]
    grid $w.body -sticky nsew
    set f [ttk::frame $c.f]
    foreach item $column_items text $column_labels {
        if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] } {
            set ::GidPriv(ColumnVisible,$category,$item) 1
        }
        set SelectColumn($item) $::GidPriv(ColumnVisible,$category,$item)
        ttk::checkbutton $f.$item -text $text -variable ${category}::SelectColumn($item)
        grid $f.$item -sticky w
    }
    AddToScrolledCanvas $w.body $f

    grid rowconfigure $w 0 -weight 1
    grid columnconfigure $w 0 -weight 1
    CreateLowerButtonsApplyClose $w [list ${category}::SelectColumnsWindowOnApply $tree $category $column_items]
}

proc Layers::SelectColumnsWindowOnApply { tree category column_items } {
    variable SelectColumn
    set changes 0
    foreach item $column_items {
        if { $::GidPriv(ColumnVisible,$category,$item) != $SelectColumn($item) } {
            set ::GidPriv(ColumnVisible,$category,$item) $SelectColumn($item)
            set changes 1
        }
    }
    if { $changes } {
        TreeSelectColumnsVisibility $tree $category $column_items
    }
}

proc Layers::SwitchWindowInsideOutside {  { w .gid.central.wlay } } {
    if { ![winfo exists $w] } {
        Layers::ChangeLayers_New $w 1
    } elseif { [winfo class $w] eq "Toplevel" } {
        Layers::CloseWindowOutside $w
        Layers::ChangeLayers_New $w 1
    } else {
        Layers::CloseWindowInside $w
        Layers::ChangeLayers_New $w 0
    }
}

proc Layers::OpenWindowOutside { w } {
    if { [winfo exists $w] } {
        Layers::CloseWindowOutside $w
    }
    if { $::GidPriv(ShowGroupsTab) } {
        InitWindow2 $w -title [_ "Layers and groups"] \
        -geometryvariable PreLayersWindowGeom \
        -initcommand Layers::ChangeLayers -ontop
    } else {
        InitWindow2 $w -title [_ "Layers"] \
        -geometryvariable PreLayersWindowGeom \
        -initcommand Layers::ChangeLayers -ontop
    }
    return $w
}

proc Layers::CloseWindowOutside { w } {
    destroy $w
}


########################
#   This procedure take account new panedwindow GidPriv(pwCentral) to manage several windows
########################
proc Layers::CloseWindowInside { p } {
    global GidPriv

    if {![winfo exists $p]} return

    #######
    # pwCentral is a panedwindow to manage $grWindow with other windows in GiD
    if {![info exists GidPriv(pwCentral)]} {
        destroy $p
        return
    }
    #######

    bind $p <Configure> ""
    Layers::WriteLayerGeomToVar $p NONE PreLayersWindowGeom Layers::ChangeLayers


    if { [winfo exists $GidPriv(pwCentral)] } {
        if { [winfo class $GidPriv(pwCentral)] in "Panedwindow TPanedwindow" } {
            set panes [$GidPriv(pwCentral) panes]
            set _pos [lsearch $panes $p]
            if {$_pos >= 0} {
                if { [llength $panes] == 2 } {
                    set oldwin [lreplace $panes $_pos $_pos]
                    set mg [winfo manager $GidPriv(pwCentral)]
                    set mginfo [$mg info $GidPriv(pwCentral)]
                    $GidPriv(pwCentral) forget $p
                    $mg $oldwin {*}$mginfo
                    update idletasks
                    destroy $GidPriv(pwCentral)
                    unset GidPriv(pwCentral)
                }
            }
        }
    } else {
        unset GidPriv(pwCentral)
    }
    destroy $p
}

########################
#   Procedure to open window inside, include creation of panedwindow to manage several windows
########################
proc Layers::OpenWindowInside { w } {
    global GidPriv

    # first check if main window has a geometry manager ( grid, pack, place or ...)
    # if not, embedded window can not be placed inside
    set grWindow [GidUtils::GetMainDrawAreaWidget]
    if { [winfo manager $grWindow] == ""} {
        return ""
    }

    if { [winfo exists $w] } {
        Layers::CloseWindowInside $w
    }

    ttk::frame $w

    set focus [focus]

    #######
    # pwCentral is a panedwindow to manage $grWindow with other windows in GiD
    if {![info exists GidPriv(pwCentral)]} {
        set GidPriv(pwCentral) .gid.central.pwCentral
    }
    #######

    if {![winfo exists $GidPriv(pwCentral)]} {
        set mg [winfo manager $grWindow]
        set mginfo [$mg info $grWindow]
        $mg forget $grWindow
        panedwindow $GidPriv(pwCentral) -borderwidth 0 -showhandle 0 -sashpad 1 -sashrelief sunken -opaqueresize 0
        #ttk::panedwindow $GidPriv(pwCentral) -orient horizontal
        $mg $GidPriv(pwCentral) {*}$mginfo
        $GidPriv(pwCentral) add $grWindow -stretch always -minsize 200
        $GidPriv(pwCentral) add $w -stretch never -minsize 200 ;#e.g. the layers window, stretch only the main draw window
        raise $grWindow
        if {$grWindow == ".gid.central.wins"} {
            raise .gid.central.s
        }
        raise $w
    } else {
        set panes [$GidPriv(pwCentral) panes]
        set maxpane [expr [llength $panes]-1]
        # Are there any window after grWindow?
        if {[lsearch $panes $grWindow] < $maxpane} {
            Layers::CloseWindowInside $w
            Layers::ChangeLayers_New $w 0
        } else {
            $GidPriv(pwCentral) add $w -after $grWindow -stretch never -minsize 200
        }
    }

    if { $focus ne "" } {
        update
        catch { focus -force $focus }
    }
    return $w
}

#to be invoked automatically when GidPriv(ShowGroupsTab) changes
proc Layers::UpdateGroupsTab { } {
    if { [GidUtils::ExistsWindow LAYER] } {
        #GidUtils::CloseWindow LAYER
        #GidUtils::OpenWindow LAYER
        set nb .gid.central.wlay.body.nb
        set f_groups $nb.f_groups
        if { $::GidPriv(ShowGroupsTab) == 0 } {
            if { [winfo exists $nb] && [lsearch [$nb tabs] $f_groups] != -1 } {
                $nb forget $f_groups
            } else {
                #to force rebuild, without any notebook
                GidUtils::CloseWindow LAYER
                GidUtils::OpenWindow LAYER
            }
        } else {
            if { [winfo exists $nb] && [lsearch [$nb tabs] $f_groups] == -1 } {
                $nb add $f_groups -text [_ "Groups"] -underline 0
            } else {
                #to force rebuild creating notebook
                GidUtils::CloseWindow LAYER
                GidUtils::OpenWindow LAYER
            }
        }
    }
}
